package mcfall.math;

/**
 * <p>
 * The Ray class represents the geometric notion of a ray, which has a starting point and
 * a direction associated with it.  This implementation allows the ray to be treated like
 * a parametrically described line.  If <i>t</i> is the parameter for the line, then the line
 * is located at the ray's starting point when <i>t</i>=0, and at a user defined "ending" point
 * when <i>t</i>=1.
 * </p>
 * 
 * @author mcfall
 */
public class Ray {
	
	/** The Ray's starting point  *. */
	private Point start;
	
	/** The Ray's direction  *. */
	private Vector direction;
	
	/**
	 * Construct a Ray object whose starting point is at <i>start</i>, and moves in the
	 * direction from <i>start</i> to <i>end</i>.  Evaluating the parameteric representation
	 * of the ray will yield the point <i>start</i> when the parameter is 0, and <i>end</i>
	 * when the parameter is 1.
	 * 
	 * @param start the starting point for the ray
	 * @param end the point at which the parameteric representation of the ray will be at time 0
	 */
	public Ray (Point start, Point end) {
		setStart (start);
		direction = new ColumnVector (4);
		direction.setValueAt(1, end.getX()-start.getX());
		direction.setValueAt(2, end.getY()-start.getY());
		direction.setValueAt(3, end.getZ()-start.getZ());
		direction.setValueAt(4, 0);
	}
	
	/**
	 * Construct a Ray object whose starting point is at <i>start</i> and moves in the
	 * direction <i>direction</i>.  Evaluating the parametric representation of the ray
	 * will yield the point <i>start</i> when the parameter is 0, and will yield the
	 * point start.moveBy (direction) when the parameter is 1
	 * 
	 * @param start the start
	 * @param direction the direction
	 */
	public Ray (Point start, Vector direction) {		
		this (start, start.add(direction));
	}
	
	/**
	 * Constructs a ray whose starting point and direction are identical to the specified ray.
	 * 
	 * @param source the ray to copy
	 */
	public Ray (Ray source) {
		start = new Point (source.start.getX(), source.start.getY(), source.start.getZ());
		direction = source.direction.duplicate();
	}
	
	/**
	 * Returns the location of the ray for a given value of the parameter <i>t</i>.
	 * 
	 * @param t the value of the parameter for the Ray, in the range 0 to infinity.  Negative
	 * values are t <b>are</b> allowed, but line in the opposite direction of the ray's direction
	 * 
	 * @return a Point along the ray
	 */
	public Point pointAt (double t) {
		Vector shiftVector = new ColumnVector(4);
		for (int index = 1; index <= 4; index++) {
			shiftVector.setValueAt(index, t*direction.getValueAt(index));
		}		
		return start.add(shiftVector);
	}
	
	/**
	 * Creates a Ray that is the result of transforming this ray by the specified matrix
	 * A Ray is transformed by applying the transformation to both the starting
	 * point and the direction of the Ray.
	 * 
	 * @param transformation a Matrix representing the transformation to use
	 * 
	 * @return a new Ray that is the result of transforming this ray by <i>transformation</i>
	 */
	public Ray transform (Matrix transformation) {
		//  TODO  Talk with Herb about what the best way to do this would be
		//  Should it be done as a method in this class, or a method in Point,
		//  or a method in matrix?
		try {
			Point newStart = Point.fromColumnMatrix(getStart().premultiply(transformation));
			Vector newDirection = Vector.fromColumnMatrix(getDirection().premultiply(transformation));
			Ray returnRay = new Ray (newStart, newDirection);
			return returnRay;
		}
		catch (IncompatibleMatrixException e) {
			//  This should never happen, so we wrap it in a runtime exception
			//  and rethrow it
			throw new RuntimeException (e);
		}		
	}

	/**
	 * Gets the direction.
	 * 
	 * @return the direction
	 */
	public Vector getDirection() {
		return direction;
	}

	/**
	 * Sets the direction.
	 * 
	 * @param direction the new direction
	 */
	public void setDirection(Vector direction) {
		this.direction = direction;
	}

	/**
	 * Gets the start point.
	 * 
	 * @return the starting point
	 */
	public Point getStart() {
		return start;
	}

	/**
	 * Sets the starting point.
	 * 
	 * @param start the new start point
	 */
	public void setStart(Point start) {
		this.start = start;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuffer buffer = new StringBuffer (32);
		buffer.append (getStart().toString());
		buffer.append (" + ");
		buffer.append (getDirection().toString());
		buffer.append ("t");
		return buffer.toString();
	}
	
	
}
